%% The following script will only run if it has access to the following
clear;
addpath('./EM_sources/');
addpath('./Scatterer_discretization/');
addpath('./CDDA_solution/');
addpath('./Scattering_calculations/');
addpath('./Constants/');

%% single sphere example (single wavelength)

% This script illustrates how the CDDA functions must be employed in order 
% to do scattering calculations when only one scatterer is present. 
% The script is clearly separated en three parts. The first one consist 
% in the definition of the diferent parameters that describe the scatterer 
% and the illumination source. The second consist in solving the coupling 
% of the dipolar units. The third is the part where scattering calculations
% are done by means of the solution obtained from the coupled equations.

% The problem treated here is the following: scattering produced by a
% sphere with circular birrefringence illuminated with a plane wave. The
% circular birrefringence is quantified by the mixed magnetic-electric
% susceptibility chi, i.e., the refractive index (RI) for right circularly
% polarized ligth (CPL) is nR = n0 + chi, while for left CPL is nL = n0 -
% chi.

% The results obtained here can be directly compared with an analitic
% solution given for example at [1] like it has been already done at [2]
% with other parameters.

% [1] C. F. Bohren and D. R. Huffman, eds., Absorption and Scattering of 
% Light by Small Particles (Wiley-VCH Verlag GmbH, Weinheim, Germany, 
% 1998).

% [2]Rosales, S. A., Albella, P., González, F., Gutierrez, Y., & Moreno, F. 
% (2021). CDDA: extension and analysis of the discrete dipole approximation 
% for chiral systems. Optics Express, 29(19), 30020-30034.


%%%%%%%%%                                        %%%%%%%%%
%%%%%%%%% Definition of scatterer and plane wave %%%%%%%%%
%%%%%%%%%                                        %%%%%%%%%

%%%% Sphere parameters %%%%
r      = 80;      % Radius  
nsc    = 1.4;     % RI for unpolarized light (mean).
chi    = 1e-4;    % nR = nsc + chi, nL = nsc - chi. Due to chirality of the
                  % sphere material
Ns     = 200;     % Number of dipoles desired for the discretization. The 
                  % larger this number the smaller is the error, but longer
                  % the calculation
pos0s  = [0,0,0]; % Position of the center of mass of the sphere
nm     = 1.3;     % Surrounding media RI.

%%%% Plane wave parameters %%%%
E      = 1;               % Amplitude
theta  = 0;               % First angle of propagation direction definition
phi    = 0;               % Second angle of propagation direct. def.
                          % If theta = 0, the direction of propagation is
                          % [0,0,1] (cartesian) whatever the value of phi
                          % is.
                          % The plane wave satisfies the null-divergence
                          % property i.e. k.E = 0.
                          
angle  = pi/4;            % Polarization orientation. The case 
                          % theta = 0, angle = 0 is the x polarization. 
phase  = pi/2;            % Phase diference between the two ortogonal
                          % components of the electric field.
lambda = 300;             % Wavelength
k0     = 2*pi*nm/lambda;  % Wavenumber


%%%%%%%%%                                              %%%%%%%%%
%%%%%%%%% Generation and solution of coupled equations %%%%%%%%%
%%%%%%%%%                                              %%%%%%%%%

tic

%%%% Discretization of the scatterers %%%%

[pos,a]          = spherical_distribution_dipoles(r,Ns,pos0s);
[alphae, alphah, G] = polarizabilities(nsc^2, nm^2, 1, 1, chi, a, k0);

%%%% Definition of the illumination %%%%
[EH01,vk]  = plane_wave(E,angle, phase,theta,phi);  %% Right CPL
[EH02]     = plane_wave(E,angle,-phase,theta,phi);  %% Left CPL


%%%% Total incident field in each dipole (DDA solution) %%%%
[EH1] = CDDA_solution(pos,alphae,alphah,G,EH01,k0,nm^2);
[EH2] = CDDA_solution(pos,alphae,alphah,G,EH02,k0,nm^2);

%%%%%%%%%                         %%%%%%%%%
%%%%%%%%% Scattering calculations %%%%%%%%%
%%%%%%%%%                         %%%%%%%%%


zc  = 1e6;
rff = zc*vk.'; % Position arbitrarily far from the sphere in the foward
               % direction
               
E1  = far_fields(pos,alphae,alphah,G,EH1,k0,rff,nm^2); % E-far field
E2  = far_fields(pos,alphae,alphah,G,EH2,k0,rff,nm^2); % E-far field

Eff1     = norm(E1)^2;
Eff2     = norm(E2)^2;

rnf = r*vk.'; % Position on the surface of the sphere in the foward
              % direction
              
E1  = near_fields(pos,alphae,alphah,G,EH1,EH01,k0,rnf,nm^2,1); % E-near field
E2  = near_fields(pos,alphae,alphah,G,EH2,EH02,k0,rnf,nm^2,1); % E-near field

Enf1     = norm(E1)^2;
Enf2     = norm(E2)^2;


%%% Cross sections %%%
[Cextr,Cscar,Cabsr] = cross_sections(pos,alphae,alphah,G,EH1,EH01,k0,nm^2);
[Cextl,Cscal,Cabsl] = cross_sections(pos,alphae,alphah,G,EH2,EH02,k0,nm^2);


toc 


%% single sphere example (multiple wavelengths)

%%%%%%%%%                                        %%%%%%%%%
%%%%%%%%% Definition of scatterer and plane wave %%%%%%%%%
%%%%%%%%%                                        %%%%%%%%%

%%%% Sphere parameters %%%%
r      = 80;      % Radius  
nsc    = 1.4;     % RI for unpolarized light (mean).
chi    = 1e-4;    % nR = nsc + chi, nL = nsc - chi. Due to chirality of the
                  % sphere material
Ns     = 427;      % Number of dipoles desired for the discretization. The 
                  % larger this number the smaller is the error, bu longer
                  % the calculation
pos0s  = [0,0,0]; % Position of the center of mass of the sphere
nm     = 1.3;     % Surrounding media RI.

%%%% Plane wave parameters %%%%
E      = 1;               % Amplitude
theta  = 0;               % First angle of propagation direction definition
phi    = 0;               % Second angle of propagation direct. def.
                          % If theta = 0, the direction of propagation is
                          % [0,0,1] (cartesian) whatever the value of phi
                          % is.
                          % The plane wave satisfies the null-divergence
                          % property i.e. k.E = 0.
                          
angle  = pi/4;            % Polarization orientation. The case 
                          % theta = 0, angle = 0 is the x polarization. 
phase  = pi/2;            % Phase diference between the two ortogonal
                          % components of the electric field.
lambda = 250:5:450;       % Wavelengths
k0     = 2*pi*nm./lambda; % Wavenumbers


%%%%%%%%%                                              %%%%%%%%%
%%%%%%%%% Generation and solution of coupled equations %%%%%%%%%
%%%%%%%%%                                              %%%%%%%%%

tic

%%%% Discretization of the scatterers %%%%
[pos,a]             = spherical_distribution_dipoles(r,Ns,pos0s);
[alphae, alphah, G] = polarizabilities(nsc^2, nm^2, 1, 1, chi, a, k0);

%%%% Definition of the illumination %%%%
[EH01,vk]  = plane_wave(E,angle, phase,theta,phi);  %% Right CPL
[EH02]     = plane_wave(E,angle,-phase,theta,phi);  %% Left CPL


%%%% Total incident field in each dipole (DDA solution) %%%%
[EH1] = CDDA_solution(pos,alphae,alphah,G,EH01,k0,nm^2);
[EH2] = CDDA_solution(pos,alphae,alphah,G,EH02,k0,nm^2);


%%%%%%%%%                         %%%%%%%%%
%%%%%%%%% Scattering calculations %%%%%%%%%
%%%%%%%%%                         %%%%%%%%%


zc  = 1e6;
rff = zc*vk.'; % Position arbitrarily far from the sphere in the foward
               % direction
E1  = far_fields(pos,alphae,alphah,G,EH1,k0,rff,nm^2); % E-far field
E2  = far_fields(pos,alphae,alphah,G,EH2,k0,rff,nm^2); % E-far field

Effr     = abs(E1).^2;
Effl     = abs(E2).^2;

rnf = zc*vk.'; % Position on the surface of the sphere in the foward
              % direction
E1  = near_fields(pos,alphae,alphah,G,EH1,EH01,k0,rnf,nm^2,1); % E-near field
E2  = near_fields(pos,alphae,alphah,G,EH2,EH02,k0,rnf,nm^2,1); % E-near field

Enfr     = sum(abs(E1.').^2).';
Enfl     = sum(abs(E2.').^2).';


%%% Cross sections %%%
[Cextr,Cscar,Cabsr] = cross_sections(pos,alphae,alphah,G,EH1,EH01,k0,nm^2);
[Cextl,Cscal,Cabsl] = cross_sections(pos,alphae,alphah,G,EH2,EH02,k0,nm^2);


toc 

